<?php

namespace InstantUploader;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class Core
 * @since  1.0.0
 */
class Core {

	/**
	 * Instance.
	 * Holds the plugin instance.
	 * @since  1.0.0
	 * @access public
	 * @static
	 * @var $instance Core.
	 */
	public static $instance = null;
	private $admin;
	private $cron;
	private $user_activity;

	/**
	 * Instance.
	 * @return Core|null
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();

			/**
			 * instantup loaded.
			 * @since 1.0.0
			 */
			do_action( 'instantup_load' );
		}

		return self::$instance;
	}

	/**
	 * Constructor loads API functions, defines paths and adds required wp actions
	 * @since  1.0
	 */
	private function __construct() {

		/**
		 * Include autoloader.
		 */
		require __DIR__ . '/autoloader.php';

		/**
		 * Register autoloader and add namespaces.
		 */
		$this->register_autoloader();

		/**
		 * Init components.
		 */
		$this->admin         = new Admin();
		$this->cron          = new Cron();
		$this->user_activity = new User_Activity();
		$this->turn_on_off();
		$this->init_components();
	}

	/**
	 * Register autoloader.
	 * @since  1.0.0
	 * @access private
	 */
	private function register_autoloader() {

		/**
		 *  Get the autoloader.
		 * */
		$autoloader = new Autoloader();

		/**
		 *  Register the autoloader.
		 * */
		$autoloader->register();

		/**
		 * Register the base directories for the namespace prefix.
		 * */
		$autoloader->add_namespace( 'InstantUploader', __DIR__ );
	}

	/**
	 * Register activation and deactivation plugin.
	 * @since  1.0.0
	 * @access private
	 */
	private function turn_on_off() {
		register_activation_hook( INSTANTUP_FILE, array( $this, 'activator' ) );
		register_deactivation_hook( INSTANTUP_FILE, array( $this, 'deactivator' ) );
	}

	/**
	 * Activation plugin.
	 * @since  1.0.0
	 * @access public
	 */
	function activator() {
		$this->cron->init_cron();
		$this->admin->activation_plugin_iu();
		$this->user_activity->creating_db();

		global $wp_rewrite;
		$wp_rewrite->flush_rules( true );
		flush_rewrite_rules();
	}

	/**
	 * Deactivation plugin.
	 * @since  1.0.0
	 * @access public
	 */
	function deactivator() {
		$this->cron->terminate_cron();

		if ( $iu_post_type_slug = get_option( '_iu_custom_type_slug' ) ) {
			foreach ( $iu_post_type_slug as $item ) {
				delete_option( "_rewrite_project_slug_{$item}" );
			}
		}

		global $wp_rewrite;
		$wp_rewrite->flush_rules( true );
		flush_rewrite_rules();
	}

	/**
	 * Init instantup components.
	 * @since  1.0.0
	 * @access public
	 */
	public function init_components() {
		$this->admin->init();
		$this->cron->init();
		$this->user_activity->init();

		$SaS = new scripts_and_styles();
		$SaS->init();

		$filters = new filters_for_media();
		$filters->init();

		$templates = new Templates();
		$templates->init();

		$options = new Options();
		$options->init();

		$types = new Types();
		$types->init();

		$metaboxes = new Metaboxes();
		$metaboxes->init();

		$importer = new Importer();
		$importer->init();

		$webhooks = new Webhooks();
		$webhooks->init();

		$blocks = new Blocks();
		$blocks->init();

		$localize = new Localize();
		$localize->init();

		/**
		 * InstantUp init.
		 * @since 1.0.0
		 */
		do_action( 'instantup_init' );
	}


}

Core::instance();