<?php

namespace InstantUploader;

if ( ! defined( 'WPINC' ) ) {
	die();
}

class Importer extends Admin {
	/**
	 * Leads constructor.
	 * @since  1.0.0
	 * @access public
	 */
	public function __construct() {
		Admin::__construct();
	}

	/**
	 * Init importer.
	 * @since  1.0.0
	 * @access public
	 */
	public function init() {
		add_action( 'wp_ajax_project_import_firebase', [ $this, 'ajax_import_with_firebase' ] );
		add_action( 'edit_post', [ $this, 'export_saas_data_status_post' ], 10, 2 );
	}

	public function export_saas_data_status_post( $postid, $post ) {
		$default_display_name = 'Empty';
		$current_user_id      = get_current_user_id();

		if ( $default_user = get_user_by( 'id', $current_user_id ) ) {
			$default_display_name = property_exists( $default_user->data, 'display_name' )
				? $default_user->data->display_name
				: 'Empty';
		}
		else {
			$admins = get_users( [ 'role' => 'administrator', 'orderby' => 'ID', 'order' => 'ASC' ] );

			if ( ! empty( $admins ) ) {
				$first_admin          = $admins[0];
				$default_display_name = property_exists( $first_admin, 'display_name' )
					? $first_admin->display_name
					: 'Empty';
			}
		}

		$data_for_export = [
			'uid'            => get_post_meta( $postid, 'iu-uid', true ) ?: '',
			'title'          => $post->post_title,
			'url'            => get_permalink( $postid ),
			'accountId'      => get_option( 'iu_saas_accountId' ) ?: '',
			'contentCreator' => $default_display_name,
			'contentStatus'  => get_post_meta( $postid, 'iu-status_publish', true ) ?: '',
			'contentWord'    => $post->post_content,
			'date'           => date( 'Y-m-d' ) . 'T' . date( 'H:i:s.Z' ) . 'Z',
			'time_created'   => get_post_time( 'U', '', $postid ),
			'time_update'    => time(),
		];

		if ( $post->post_status === 'trash' ) {
			$data_for_export['status'] = 'unpublished';
			$this->export_saas_data( $data_for_export );
		}
	}

	public function export_saas_data( $data, $is_cron = false ) {
		if ( ! $is_cron ) {
			if ( empty( $data ) ) {
				$this->write_log( 'empty data', 'log-error.txt' );

				return false;
			}

			if ( isset( $data['success'] ) && $data['success'] === false ) {
				$this->write_log( $data, 'log-error.txt' );

				return false;
			}

			$api_key     = get_option( '_firebase_auth' );
			$export_data = [ 'data' => [] ];

			$export_data['data'][0]['url']            = $data['url'];
			$export_data['data'][0]['home_url']       = home_url();
			$export_data['data'][0]['uid']            = $data['uid'];
			$export_data['data'][0]['tags']           = json_encode( $data['tags'] ?? '' );
			$export_data['data'][0]['date']           = $data['date'];
			$export_data['data'][0]['title']          = $data['title'];
			$export_data['data'][0]['status']         = $data['status'];
			$export_data['data'][0]['time_update']    = $data['time_update'] ?? '';
			$export_data['data'][0]['contentType']    = $data['contentType'] ?? '';
			$export_data['data'][0]['contentWord']    = $data['contentWord'];
			$export_data['data'][0]['time_created']   = $data['time_created'];
			$export_data['data'][0]['contentStatus']  = $data['contentStatus'];
			$export_data['data'][0]['contentCreator'] = $data['contentCreator'];
			$export_data['apiKey']                    = $api_key;
			$export_data['accountId']                 = $data['accountId'];
		}

		if ( $is_cron ) {
			$export_data = $data;
		}

		$ch = curl_init( IU_CLOUDFUNCTIONS_HOST . '/app/publish' );
		curl_setopt( $ch, CURLOPT_POST, 1 );
		curl_setopt( $ch, CURLOPT_POSTFIELDS, json_encode( $export_data ) );
		curl_setopt( $ch, CURLOPT_HTTPHEADER, [ 'Content-Type:application/json' ] );
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
		curl_setopt( $ch, CURLOPT_CONNECTTIMEOUT, 20 );
		curl_setopt( $ch, CURLOPT_TIMEOUT, 20 );
		curl_exec( $ch );

		if ( curl_errno( $ch ) ) {
			$this->re_export_saas_data( $export_data );

			$error = date( 'Y-m-d H:i:s P' ) . ':__ ' . json_encode( curl_error( $ch ) );
			$this->write_log( $error, 'error-data-export.txt' );
		}

		curl_close( $ch );

		$this->write_log(
			date( 'Y-m-d H:i:s P' ) . ':__ ' . json_encode( $export_data ),
			'data-export.txt'
		);
	}

	public function re_export_saas_data( $data ) {
		if ( ! isset( $data['data'][0]['uid'] ) || empty( $data['data'][0]['uid'] ) ) {
			return;
		}

		$transient_name = "export_{$data['data'][0]['uid']}";
		$time_schedules = $this->custom_schedules();
		$time_execute   = $time_schedules[0];

		if ( $export_data = json_decode( get_transient( $transient_name ), true ) ) {
			$time_item           = ++ $export_data['time'];
			$time_execute        = $time_schedules[ $time_item ];
			$export_data['time'] = $time_item;
		}
		else {
			$export_data = [ 'data' => $data, 'time' => 0 ];
		}

		set_transient( $transient_name, json_encode( $export_data ), 25 * HOUR_IN_SECONDS );

		wp_schedule_single_event( $time_execute, self::get_type_prefix() . '_init_re_export', [
			$transient_name,
		] );
	}

	private function add_category( $category_data, $taxonomy, $parent_id = '' ) {
		$exists_cat = term_exists( $category_data['label'], $taxonomy );

		if ( is_array( $exists_cat ) ) {
			return (int) $exists_cat['term_id'];
		}

		$cat_array = [ 'cat_name' => $category_data['label'], 'taxonomy' => $taxonomy ];

		if ( $parent_id ) {
			$cat_array['category_parent'] = $parent_id;
		}

		if ( $cat_id = wp_insert_category( $cat_array ) ) {
			update_term_meta( $cat_id, 'uid', $category_data['uid'] );

			return (int) $exists_cat['term_id'];
		}
	}

	private function set_post_blog() {
		if ( $set_blog_type = get_option( '_whither_blog_type' ) ) {
			return [ 'post_type' => $set_blog_type, 'set_blog_cat' => '' ];
		}

		if ( $set_blog_cat = carbon_get_theme_option( 'whither_blog_cat' ) ) {
			if ( isset( $set_blog_cat[0]['id'] ) && ! empty( $set_blog_cat[0]['id'] ) ) {
				return [ 'post_type' => 'post', 'set_blog_cat' => (int) $set_blog_cat[0]['id'] ];
			}
		}

		return [ 'post_type' => '', 'set_blog_cat' => '' ];
	}

	private function get_title_and_content( $content ) {
		$result = [
			'ai_writeup' => false,
			'title'      => $content['title'] ?: '',
			'content'    => $content['description'] ?: '',
		];

		if ( isset( $content['writings'] ) && ! empty( $content['writings'] ) ) {
			foreach ( $content['writings'] as $item ) {
				if ( ! isset( $item['checkeds'] ) ) {
					continue;
				}

				foreach ( $item['checkeds'] as $checked ) {
					if ( $checked['type'] !== 'website' && empty( $checked['url'] ) ) {
						continue;
					}

					if ( $checked['url'] === home_url() ) {
						$result['ai_writeup'] = true;

						if ( isset( $item['title'] ) && ! empty( $item['title'] ) ) {
							$result['title'] = $item['title'];
						}

						if ( isset( $item['description'] ) && ! empty( $item['description'] ) ) {
							$result['content'] = $item['description'];
						}

						break 2;
					}
				}
			}
		}

		if ( empty( $result['title'] ) ) {
			if ( isset( $content['testimonial']['title'] ) ) {
				$result['title'] = $content['testimonial']['title'];
			}
			elseif ( isset( $content['testimonial']['customerTitle'] ) ) {
				$result['title'] = $content['testimonial']['customerTitle'];
			}
			elseif ( isset( $content['testimonial']['customerName'] ) ) {
				$result['title'] = $content['testimonial']['customerName'];
			}
			elseif (
				isset( $content['description'] ) &&
				mb_strlen( trim( $content['description'] ) ) > 5
			) {
				$result['title'] = mb_substr( trim( $content['description'] ), 0, 21 );
			}
			elseif (
				isset( $content['testimonial']['description'] ) &&
				mb_strlen( trim( $content['testimonial']['description'] ) ) > 5
			) {
				$result['title'] = mb_substr( trim( $content['testimonial']['description'] ), 0, 21 );
			}
		}

		return $result;
	}

	public function whether_is_img( $wpdb, $uid_img ) {
		$sql      = "SELECT post_id FROM $wpdb->postmeta WHERE `meta_key` = 'uid' AND `meta_value` = '{$uid_img}'";
		$my_media = $wpdb->get_row( $sql, ARRAY_A );

		return $my_media['post_id'] ?? '';
	}

	private function save_img( $project_media, $thumb = false ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/media.php';

		$src_img    = $project_media['url'];
		$mime_type  = $project_media['type'] ?? '';
		$is_allowed = [ 'ext' => false ];
		$tmp_img    = '';

		//Edit link to image (get image size 1920x1080)
		if ( ! $thumb && strpos( $project_media['url'], 'firebasestorage.googleapis.com' ) !== false ) {
			$src_img = str_replace( '_200x200.', '_1920x1080.', $project_media['url'] );

			if ( strpos( $project_media['url'], 'image%2Fthumb%2F' ) === false ) {
				$url_img   = str_replace( '/o/image%2F', '/o/image%2Fthumb%2F', $project_media['url'] );
				$end_point = strripos( $url_img, '.' );
				$src_img   = substr( $url_img, 0, $end_point );
				$src_img   .= '_1920x1080.';
				$src_img   .= substr( $url_img, $end_point + 1, - 1 );
			}
		}

		if ( empty( $mime_type ) ) {
			$tmp_img = download_url( $src_img, 30 );
			if ( ! is_wp_error( $tmp_img ) ) {
				$image_type = exif_imagetype( $tmp_img );
				$mime_type  = image_type_to_mime_type( $image_type );
			}
		}

		$is_allowed = wp_check_filetype( str_replace( '/', '.', $mime_type ) );

		if ( $is_allowed['ext'] ) {
			if ( ! $tmp_img ) {
				$tmp_img = download_url( $src_img, 30 );
			}
			if ( is_wp_error( $tmp_img ) ) {
				$tmp_img = download_url( $project_media['url'], 30 );
			}

			if ( isset( $project_media['name'] ) && ! empty( $project_media['name'] ) ) {
				if ( strpos( $project_media['name'], $is_allowed['ext'] ) !== false ) {
					$img_name = $project_media['name'];
				}
				else {
					$img_name = "{$project_media['name']}.{$is_allowed['ext']}";
				}
			}
			else {
				$link         = parse_url( $project_media['url'] );
				$img_name_raw = basename( $link['path'] );

				if ( strpos( $img_name_raw, $is_allowed['ext'] ) !== false ) {
					$img_name = $img_name_raw;
				}
				else {
					$img_name = "{$img_name_raw}.{$is_allowed['ext']}";
				}
			}

			$media_title = $project_media['mediaTitle'] ?: $project_media['name'];
			$img_name    = str_replace(
				[ '%2', '~', '!', '@', '#', '$', '%', '^', '&', '*', '=', '|', '`', '"' ],
				'-',
				$img_name
			);

			if ( empty( $img_name ) ) {
				$img_name = "{$project_media['uid']}.{$is_allowed['ext']}";
			}
			if ( empty( $media_title ) ) {
				$media_title = $project_media['uid'];
			}

			$file_array = [
				'name'     => $img_name, // ex: wp-logo.png
				'tmp_name' => $tmp_img,
				'error'    => 0,
				'size'     => filesize( $tmp_img ),
			];

			$media_data = [ 'post_title' => $media_title, 'post_mime_type' => $is_allowed['type'] ];

			if ( isset( $project_media['description'] ) && ! empty( $project_media['description'] ) ) {
				$media_data['post_content'] = $project_media['description'];
			}
			if ( isset( $project_media['caption'] ) && ! empty( $project_media['caption'] ) ) {
				$media_data['post_excerpt'] = $project_media['caption'];
			}

			$media_id = media_handle_sideload( $file_array, 0, $media_title, $media_data );
			@unlink( $tmp_img ); // delete temp file

			if ( ! is_wp_error( $media_id ) ) {
				update_post_meta( $media_id, 'uid', $project_media['uid'] );

				if ( isset( $project_media['alt'] ) && ! empty( $project_media['alt'] ) ) {
					update_post_meta( $media_id, '_wp_attachment_image_alt', $project_media['alt'] );
				}

				return $media_id;
			}

			@unlink( $file_array['tmp_name'] );
			$media_error[] = $media_id->get_error_messages(); //TODO no error is sent
			$this->write_log( $media_error, 'error_save_img.txt' );

			return false;
		}
	}

	public function import_metapost( $project, $prefix = 'iu-' ) {
		$meta_input = [];
		$exclude    = [
			'id'                           => 1,
			'url'                          => 1,
			'lock'                         => 1,
			'media'                        => 1,
			'notes'                        => 1,
			'tasks'                        => 1,
			'title'                        => 1,
			'userId'                       => 1,
			'accountId'                    => 1,
			'smartTags'                    => 1,
			'contentType'                  => 1,
			'description'                  => 1,
			'destinations'                 => 1,
			'businessTags'                 => 1,
			'neighborhood'                 => 1,
			'presentation'                 => 1,
			'datetimeStamp'                => 1,
			'businessLabels'               => 1,
			'businessServices'             => 1,
			'businessCategories'           => 1,
			'othersWhoWorkedOnThisProject' => 1,
		];

		$all_meta_name         = get_option( 'all_iu_meta_name', [] );
		$need_update_meta_name = false;

		foreach ( $project as $key => $field ) {
			if ( ! isset( $exclude[ $key ] ) ) {
				if ( strtolower( $key ) === 'address' && ! empty( $field ) ) {
					$address                           = explode( ',', $field );
					$meta_input[ $prefix . 'city' ]    = $address[0];
					$meta_input[ $prefix . 'county' ]  = $address[1];
					$meta_input[ $prefix . 'country' ] = end( $address );
					$meta_input[ $prefix . 'address' ] = $field;

					if ( count( $address ) === 4 ) {
						$meta_input[ $prefix . 'state' ] = $address[2];
					}
					continue;
				}

				if ( strtolower( $key ) === 'coords' && ! empty( $field ) ) {
					if ( is_string( $field ) ) {
						$coords = explode( ',', $field );

						if (
							isset( $coords[0], $coords[1] ) &&
							$coords[0] !== 'undefined' &&
							$coords[1] !== 'undefined'
						) {
							$meta_input[ $prefix . 'latitude' ]  = $coords[0];
							$meta_input[ $prefix . 'longitude' ] = $coords[1];
						}
					}

					if (
						is_array( $field ) &&
						isset( $field['lat'], $field['lng'] ) &&
						! empty( $field['lat'] ) &&
						! empty( $field['lng'] )
					) {
						$meta_input[ $prefix . 'latitude' ]  = $field['lat'];
						$meta_input[ $prefix . 'longitude' ] = $field['lng'];
					}

					continue;
				}

				if ( $key === 'updateDate' ) {
					$meta_input[ $prefix . 'updateDate' ] = $field['seconds'] ?? $field['_seconds'];
					continue;
				}

				if ( $key === 'createDate' ) {
					$meta_input[ $prefix . 'start_date' ] = $field['seconds'] ?? $field['_seconds'];
					continue;
				}

				if ( $key === 'seo' ) {
					$meta_input[ $prefix . 'seo_title' ]       = $field['title'] ?? '';
					$meta_input[ $prefix . 'seo_description' ] = $field['description'] ?? '';
					continue;
				}

				if ( $key === 'status' ) {
					$meta_input[ $prefix . 'status_content' ]  = $field['content'] ?? '';
					$meta_input[ $prefix . 'status_publish' ]  = $field['publish'] ?? '';
					$meta_input[ $prefix . 'status_service' ]  = $field['service'] ?? '';
					$meta_input[ $prefix . 'status_featured' ] = $field['featured'] ?? '';
					continue;
				}

				if ( $key === 'testimonial' ) {
					$meta_input[ $prefix . 'testimonial_customerName' ]  =
						$field['customerName'] ?? '';
					$meta_input[ $prefix . 'testimonial_customerTitle' ] =
						$field['customerTitle'] ?? '';

					$meta_input[ $prefix . 'testimonial_des' ]    = $field['description'] ?? '';
					$meta_input[ $prefix . 'testimonial_rating' ] = $field['rating'] ?? '';
					$meta_input[ $prefix . 'testimonial_title' ]  = $field['title'] ?? '';

					$this->create_page_testimonials();
					continue;
				}

				$meta_input[ $prefix . $key ] = $field;

				if ( ! $all_meta_name[ $prefix . $key ] ) {
					$need_update_meta_name           = true;
					$all_meta_name[ $prefix . $key ] = ucwords( str_replace( '_', ' ', $key ) );
				}
			}
		}

		if ( $need_update_meta_name ) {
			update_option( 'all_iu_meta_name', $all_meta_name, false );
		}

		return $meta_input;
	}

	public function import_type_posts( $type_raw, $single = false, $decode = true ) {
		$type_prefix = self::get_type_prefix();
		$type_arr    = get_option( '_iu_custom_type_posts' ) ?: [];
		$slug_arr    = get_option( '_iu_custom_type_slug' ) ?: [];

		if ( $decode ) {
			$contType = json_decode( stripcslashes( $type_raw ), true );
		}
		else {
			$contType = $type_raw;
		}

		if ( ! $single ) {
			foreach ( $contType as $type ) {
				$part_slug = strtolower( str_replace( [ '-', ' ' ], '_', $type['label'] ) );
				$slug      = $this->cleanup_post_type_slug( $type_prefix . '_' . $part_slug );

				if ( strlen( $slug ) >= 20 ) {
					$slug = $part_slug;
				}

				foreach ( $type_arr as $item ) {
					if ( $slug === $item['slug'] ) {
						continue 2;
					}
				}

				$type_arr[] = [ 'id' => $type['id'], 'name' => $type['label'], 'slug' => $slug ];

				$slug_arr[] = $slug;
			}
		}
		else {
			$part_slug = strtolower( str_replace( [ '-', ' ' ], '_', $contType['name'] ) );
			$slug      = $this->cleanup_post_type_slug( $part_slug );
			$slug      = $type_prefix . '_' . $slug;
			$allowed   = true;

			foreach ( $type_arr as $item ) {
				if ( $slug === $item['slug'] ) {
					$allowed = false;
				}
			}

			if ( $allowed ) {
				$type_arr[] = [
					'id'                      => $contType['id'],
					'name'                    => $contType['name'],
					'
					slug' => $slug,
				];

				$slug_arr[] = $slug;
			}
		}

		if ( ! get_option( '_template_for_content' ) ) {
			$new_template = [];
			$template_id  = get_option( '_template_default_id' );

			foreach ( $slug_arr as $item ) {
				$new_template[ $item ] = $template_id;
			}

			update_option( '_template_for_content', $new_template );
			update_post_meta( $template_id, '_template_for_content', $new_template );
		}

		update_option( '_iu_custom_type_posts', $type_arr );
		update_option( '_iu_custom_type_slug', $slug_arr );

		return $type_arr;
	}

	public function import_category( $categories, $services_data, $taxonomy_cat = '' ) {
		require_once ABSPATH . '/wp-admin/includes/taxonomy.php';
		$tax_input   = [];
		$type_prefix = self::get_type_prefix();

		if ( empty( $taxonomy_cat ) ) {
			$taxonomy_cat = $type_prefix . '_categories';
		}

		if ( ! taxonomy_exists( $taxonomy_cat ) ) {
			$types = new Types();
			$types->new_taxonomy();
		}

		if ( empty( $services_data ) ) {
			foreach ( $categories as $business_category ) {
				$tax_input[] = $this->add_category( $business_category, $taxonomy_cat );
			}
		}
		else {
			foreach ( $categories as $business_category ) {
				foreach ( $business_category['services'] as $key => $cat_services ) {
					$parent_cat_id = $this->add_category( $business_category, $taxonomy_cat );
					$tax_input[]   = $parent_cat_id;

					foreach ( $services_data as $services ) {
						if ( $services['label'] === $cat_services['label'] ) {
							$tax_input[] = $this->add_category(
								$services,
								$taxonomy_cat,
								$parent_cat_id
							);
						}
					}
				}
			}
		}

		return array_keys( array_flip( $tax_input ) ); // array ids
	}

	public function import_tags( $businessLabels, $servicesLabels, $taxonomy = '' ) {
		$type_prefix = self::get_type_prefix();
		$tags_array  = [];

		if ( empty( $taxonomy ) ) {
			$taxonomy = $type_prefix . '_tags';
		}

		foreach ( $businessLabels as $this_tags ) {
			if ( $this_tags['tags'] ) {
				foreach ( $servicesLabels as $labels ) {
					if ( $labels['label'] === $this_tags['label'] ) {
						foreach ( $this_tags['tags'] as $tag ) {
							foreach ( $labels['tags'] as $labelTag ) {
								if (
									$tag['uid'] === $labelTag['uid'] &&
									isset( $labelTag['public'] ) &&
									$labelTag['public']
								) {
									$has_tag = term_exists( $tag['label'], $taxonomy );

									if ( is_array( $has_tag ) === false ) {
										$has_tag = term_exists( $tag['alias'], $taxonomy );
									}

									if ( is_array( $has_tag ) ) {
										$term_args           = [];
										$term_id             = (int) $has_tag['term_id'];
										$tags_array['ids'][] = $term_id;

										$tags_array['to_export'][] = [
											'uid'  => $tag['uid'],
											'name' => $tag['label'],
											'url'  => get_category_link( (int) $has_tag['term_id'] ),
										];

										if ( isset( $tag['alias'] ) && ! empty( $tag['alias'] ) ) {
											$term_args['name'] = $tag['alias'];
										}

										if (
											isset( $tag['description'] ) &&
											! empty( $tag['description'] )
										) {
											$term_args['description'] = $tag['description'];
										}

										if ( ! empty( $term_args ) ) {
											wp_update_term( $term_id, $taxonomy, $term_args );
										}
									}
									else {
										$args = [
											'cat_name' => $tag['alias'] ?: $tag['label'],
											'taxonomy' => $taxonomy,
										];

										if (
											isset( $tag['description'] ) &&
											! empty( $tag['description'] )
										) {
											$args['category_description'] = $tag['description'];
										}

										if ( $id = wp_insert_category( $args ) ) {
											$tags_array['to_export'][] = [
												'uid'  => $tag['uid'],
												'name' => $tag['label'],
												'url'  => get_category_link( $id ),
											];

											$tags_array['ids'][] = $id;
											update_term_meta( $id, 'uid', $tag['uid'] );
										}
									}
								}
							}
						}
					}
				}
			}
		}

		return $tags_array;
	}

	public function import_label_tags( $businessLabels, $taxonomy = '' ) {
		$type_prefix = self::get_type_prefix();
		$tags_array  = [];

		if ( empty( $taxonomy ) ) {
			$taxonomy = $type_prefix . '_label_tags';
		}

		foreach ( $businessLabels as $tags ) {
			$has_label = term_exists( $tags['label'], $taxonomy );

			if ( is_array( $has_label ) ) {
				$label_id = (int) $has_label['term_id'];
			}
			else {
				$args_label = [ 'cat_name' => $tags['label'], 'taxonomy' => $taxonomy ];

				if ( $label_id = wp_insert_category( $args_label ) ) {
					update_term_meta( $label_id, 'uid', $tags['uid'] );
				}
			}

			foreach ( $tags['tags'] as $tag ) {
				$has_tag = term_exists( $tag['label'], $taxonomy );

				if ( is_array( $has_tag ) === false ) {
					$has_tag = term_exists( $tag['alias'], $taxonomy );
				}

				if ( is_array( $has_tag ) ) {
					$term_args           = [];
					$term_id             = (int) $has_tag['term_id'];
					$tags_array['ids'][] = $term_id;

					$tags_array['to_export'][] = [
						'uid'  => $tag['uid'],
						'name' => $tag['label'],
						'url'  => get_category_link( (int) $has_tag['term_id'] ),
					];

					if ( isset( $tag['alias'] ) && ! empty( $tag['alias'] ) ) {
						$term_args['name'] = $tag['alias'];
					}

					if ( isset( $tag['description'] ) && ! empty( $tag['description'] ) ) {
						$term_args['description'] = $tag['description'];
					}

					if ( ! empty( $term_args ) ) {
						wp_update_term( $term_id, $taxonomy, $term_args );
					}
				}
				else {
					$args = [
						'cat_name'        => $tag['alias'] ?: $tag['label'],
						'category_parent' => $label_id,
						'taxonomy'        => $taxonomy,
					];

					if ( isset( $tag['description'] ) && ! empty( $tag['description'] ) ) {
						$args['category_description'] = $tag['description'];
					}

					if ( $id = wp_insert_category( $args ) ) {
						$tags_array['to_export'][] = [
							'uid'  => $tag['uid'],
							'name' => $tag['label'],
							'url'  => get_category_link( $id ),
						];

						$tags_array['ids'][] = $id;
						update_term_meta( $id, 'uid', $tag['uid'] );
					}
				}
			}
		}

		return $tags_array;
	}

	public function import_smart_tags( $smart_tags, $taxonomy = '' ) {
		$type_prefix = self::get_type_prefix();
		$tags_array  = [];

		if ( empty( $taxonomy ) ) {
			$taxonomy = $type_prefix . '_tags';
		}

		foreach ( $smart_tags as $tag ) {
			$has_tag = term_exists( $tag['value'], $taxonomy );

			if ( is_array( $has_tag ) === false ) {
				$has_tag = term_exists( $tag['alias'], $taxonomy );
			}

			if ( is_array( $has_tag ) ) {
				$term_args    = [];
				$term_id      = (int) $has_tag['term_id'];
				$tags_array[] = $term_id;

				if ( isset( $tag['alias'] ) && ! empty( $tag['alias'] ) ) {
					$term_args['name'] = $tag['alias'];
				}

				if ( isset( $tag['description'] ) && ! empty( $tag['description'] ) ) {
					$term_args['description'] = $tag['description'];
				}

				if ( ! empty( $term_args ) ) {
					wp_update_term( $term_id, $taxonomy, $term_args );
				}
			}
			else {
				$args = [
					'cat_name' => $tag['alias'] ?: $tag['value'], //'cat_name' => $tag['value'],
					'taxonomy' => $taxonomy,
				];

				if ( isset( $tag['description'] ) && ! empty( $tag['description'] ) ) {
					$args['category_description'] = $tag['description'];
				}

				if ( $id = wp_insert_category( $args ) ) {
					$tags_array[] = $id;
					update_term_meta( $id, 'uid', $tag['id'] );
					update_term_meta( $id, 'smart_tag', 'yes' );
				}
			}
		}

		return $tags_array;
	}

	public function import_thumb_video( $wpdb, $post_id, $testimonials ) {
		$video    = $testimonials['video'];
		$thumb_id = $this->whether_is_img( $wpdb, $video['uid'] . '_thumb' );

		if ( empty( $thumb_id ) ) {
			$thumb_title = $video['mediaTitle'] ?: $video['uid'];

			$thumb_arr = [
				'uid'        => $video['uid'] . '_thumb',
				'url'        => $video['thumb'],
				'mediaTitle' => 'thumb for ' . $thumb_title,
			];

			$thumb_id = $this->save_img( $thumb_arr );
		}

		if ( ! empty( $video['uid'] ) && ! empty( $thumb_id ) ) {
			$prefix                          = self::get_metabox_prefix();
			$testimonials_media_arr          = get_post_meta( $post_id, $prefix . 'testimonial_media', true );
			$testimonials_media_arr['cover'] = $thumb_id;

			update_post_meta( $post_id, $prefix . 'testimonial_media', $testimonials_media_arr );
		}
	}

	public function import_media( $wpdb, $post_id, $attachments, $testimonials, $presentation ) {
		require_once ABSPATH . 'wp-admin/includes/file.php';
		$prefix                 = self::get_metabox_prefix();
		$number_img             = 1;
		$item                   = 0;
		$set_thumbnail_post     = true;
		$gallery_arr            = [];
		$testimonials_media_arr = [];
		$presentation_media_arr = [];
		$gallery_images_limit   = get_option( '_iu_default_gallery_images' ) ?: 15;
		$how_many_img           = (int) ceil( count( $attachments ) / $gallery_images_limit ) ?: 1;

		if ( isset( $presentation['cover']['uid'] ) && ! empty( $presentation['cover']['uid'] ) ) {
			$p_cover_id = $this->whether_is_img( $wpdb, $presentation['cover']['uid'] );

			if ( empty( $p_cover_id ) ) {
				$p_cover_id = $this->save_img( $presentation['cover'] );
			}

			if ( ! empty( $p_cover_id ) ) {
				if ( wp_attachment_is_image( $p_cover_id ) ) {
					$set_thumbnail_post = false;
					update_post_meta( $post_id, '_thumbnail_id', $p_cover_id );
				}
				$presentation_media_arr['cover'] = $p_cover_id;
				$gallery_arr[]                   = [
					'time' => $presentation['cover']['createDate'],
					'id'   => (int) $p_cover_id,
				];
			}
		}
		if ( isset( $presentation['after']['uid'] ) && ! empty( $presentation['after']['uid'] ) ) {
			$p_after_id = $this->whether_is_img( $wpdb, $presentation['after']['uid'] );

			if ( empty( $p_after_id ) ) {
				$p_after_id = $this->save_img( $presentation['after'] );
			}

			if ( ! empty( $p_after_id ) ) {
				if ( wp_attachment_is_image( $p_after_id ) ) {
					$set_thumbnail_post = false;
					update_post_meta( $post_id, '_thumbnail_id', $p_after_id );
				}
				$presentation_media_arr['after'] = $p_after_id;

				if ( $presentation['cover']['uid'] !== $presentation['after']['uid'] ) {
					$gallery_arr[] = [
						'time' => $presentation['after']['createDate'],
						'id'   => (int) $p_after_id,
					];
				}
			}
		}
		if ( isset( $presentation['before']['uid'] ) && ! empty( $presentation['before']['uid'] ) ) {
			$p_before_id = $this->whether_is_img( $wpdb, $presentation['before']['uid'] );

			if ( empty( $p_before_id ) ) {
				$p_before_id = $this->save_img( $presentation['before'] );
			}

			if ( ! empty( $p_before_id ) ) {
				$presentation_media_arr['before'] = $p_before_id;

				if ( $presentation['cover']['uid'] !== $presentation['before']['uid'] ) {
					$gallery_arr[] = [
						'time' => $presentation['before']['createDate'],
						'id'   => (int) $p_before_id,
					];
				}
			}
		}
		if ( isset( $testimonials['customerPhoto']['uid'] ) && ! empty( $testimonials['customerPhoto']['uid'] ) ) {
			$customerPhoto_id = $this->whether_is_img( $wpdb, $testimonials['customerPhoto']['uid'] );

			if ( empty( $customerPhoto_id ) ) {
				$customerPhoto_id = $this->save_img( $testimonials['customerPhoto'] );
			}

			if ( ! empty( $customerPhoto_id ) ) {
				$testimonials_media_arr['customerPhoto'] = $customerPhoto_id;
			}
		}
		if ( isset( $testimonials['cover']['uid'] ) && ! empty( $testimonials['cover']['uid'] ) ) {
			$t_cover_id = $this->whether_is_img( $wpdb, $testimonials['cover']['uid'] );

			if ( empty( $t_cover_id ) ) {
				$t_cover_id = $this->save_img( $testimonials['cover'] );
			}

			if ( ! empty( $t_cover_id ) ) {
				$testimonials_media_arr['cover'] = $t_cover_id;

				if ( $testimonials['cover']['uid'] !== $presentation['cover']['uid'] ) {
					$gallery_arr[] = [
						'time' => $testimonials['cover']['createDate'],
						'id'   => (int) $t_cover_id,
					];
				}
			}
		}

		foreach ( $attachments as $i => $project_media ) {
			$thumb_id = '';

			if ( $project_media['uid'] ) {
				$media_id = $this->whether_is_img( $wpdb, $project_media['uid'] );

				if ( empty( $media_id ) ) {
					$media_id = $this->save_img( $project_media );
				}
			}
			else {
				$media_id = $this->save_img( $project_media );
			}

			if ( empty( $media_id ) ) {
				continue;
			}

			if ( strpos( $project_media['type'], 'video' ) !== false ) {
				$thumb_id = $this->whether_is_img( $wpdb, $project_media['uid'] . '_thumb' );

				if ( empty( $thumb_id ) ) {
					$thumb_title = $project_media['mediaTitle'] ?: $project_media['uid'];
					$parse_thumb = parse_url( $project_media['thumb'] );

					$thumb_arr = [
						'uid'        => $project_media['uid'] . '_thumb',
						'url'        => $project_media['thumb'],
						'mediaTitle' => 'thumb for ' . $thumb_title,
						'type'       => basename( $parse_thumb['path'] ),
					];

					$thumb_id = $this->save_img( $thumb_arr, true );
					update_post_meta( $media_id, $prefix . 'thumb', $thumb_id );
				}
			}

			if ( $set_thumbnail_post && ( wp_attachment_is_image( $media_id ) || $thumb_id ) ) {
				$set_thumbnail_post = false;

				if ( $thumb_id ) {
					update_post_meta( $post_id, '_thumbnail_id', $thumb_id );
				}
				else {
					update_post_meta( $post_id, '_thumbnail_id', $media_id );
				}
			}

			if (
				( isset( $testimonials['cover']['uid'] ) && $project_media['uid'] === $testimonials['cover']['uid'] ) ||
				( isset( $testimonials['customerPhoto']['uid'] ) && $project_media['uid'] === $testimonials['customerPhoto']['uid'] ) ||
				( isset( $presentation['cover']['uid'] ) && $project_media['uid'] === $presentation['cover']['uid'] ) ||
				( isset( $presentation['after']['uid'] ) && $project_media['uid'] === $presentation['after']['uid'] ) ||
				( isset( $presentation['before']['uid'] ) && $project_media['uid'] === $presentation['before']['uid'] )
			) {
				unset( $attachments[ $i ] );
				continue;
			}

			if ( isset( $testimonials ) && array_key_exists( 'video', $testimonials ) && is_null( $testimonials['video'] ) ) {
				$testimonials_media_arr['video'] = '';
				//unset( $attachments[ $i ] );
			}

			if ( isset( $testimonials['video']['uid'] ) && $project_media['uid'] === $testimonials['video']['uid'] ) {
				$testimonials_media_arr['video'] = $media_id;
				unset( $attachments[ $i ] );
				continue;
			}

			if ( isset( $presentation['gallery'] ) && ! empty( $presentation['gallery'] ) ) {
				foreach ( $presentation['gallery'] as $img ) {
					if ( $project_media['uid'] === $img['uid'] ) {
						$gallery_arr[] = [
							'time' => $img['createDate'],
							'id'   => (int) $media_id,
						];
					}
				}
			}
			else {
				if ( $how_many_img === 1 ) {
					$gallery_arr[] = [
						'time' => $project_media['createDate'],
						'id'   => (int) $media_id,
					];
				}
				else {
					if ( $item === 0 ) {
						$gallery_arr[] = [
							'time' => $project_media['createDate'],
							'id'   => (int) $media_id,
						];
					}
					else {
						if ( $item === $number_img ) {
							$number_img    += $how_many_img;
							$gallery_arr[] = [
								'time' => $project_media['createDate'],
								'id'   => (int) $media_id,
							];
						}
					}
				}
			}

			$item ++;
		} //end foreach

		if ( ! empty( $testimonials_media_arr ) ) {
			update_post_meta( $post_id, $prefix . 'testimonial_media', $testimonials_media_arr );
		}

		if ( ! empty( $presentation_media_arr ) ) {
			update_post_meta( $post_id, $prefix . 'presentation', $presentation_media_arr );
		}

		arsort( $gallery_arr );
		update_post_meta( $post_id, $prefix . 'pic_for_post', $gallery_arr );
	}

	public function ajax_import_with_firebase() {
		if ( empty( $_POST['data'] ) && empty( $_POST['contType'] ) ) {
			wp_send_json( [ 'success ' => false, 'message' => 'empty data' ] );
		}

		$was_suspended = wp_suspend_cache_addition();
		wp_suspend_cache_addition( true );

		global $wpdb;

		$content         = json_decode( stripcslashes( $_POST['data'] ), true );
		$prefix          = self::get_metabox_prefix(); //iu-
		$meta_key        = $prefix . 'uid';
		$sql             = $wpdb->prepare(
			"SELECT post_id FROM $wpdb->postmeta WHERE `meta_key` = %s AND meta_value = %s",
			$meta_key,
			$content['uid']
		);
		$project_id      = $wpdb->get_row( $sql, ARRAY_A );
		$id_post         = $project_id['post_id'];
		$data_for_export = [];

		$this->import_type_posts( $_POST['contType'] );

		if ( isset( $_POST['skipProject'] ) && $_POST['skipProject'] ) {
			if ( ! $id_post ) {
				$data_for_export = $this->update_content( $content ); //create content
			}
		}
		else {
			if ( $id_post ) {
				$data_for_export = $this->update_content( $content, $id_post ); // update content
			}
			else {
				$data_for_export = $this->update_content( $content ); //create content
			}
		}

		if ( $data_for_export ) {
			$this->export_saas_data( $data_for_export );
		}

		wp_suspend_cache_addition( $was_suspended );
		wp_send_json( $data_for_export );
	}

	public function webhook_import( $project, $content_type = '' ) {
		global $wpdb;

		$prefix   = self::get_metabox_prefix(); //iu-
		$meta_key = $prefix . 'uid';
		$content  = $project['content'];

		if ( $project['triggerType'] === 'add_site' ) {
			$response            = [];
			$response['apiKey']  = $project['apiKey'];
			$response['message'] = 'Website is successfully connected';
			$response['status']  = true;

			status_header( 200 );
			exit( json_encode( $response ) );
		}

		if ( $project['triggerType'] === 'description_is_updated' ) {
			if ( $content['businessCategoryId'] ) {
				$id = $content['businessCategoryId'];
			}
			elseif ( $content['serviceId'] ) {
				$id = $content['serviceId'];
			}
			elseif ( $content['tagId'] ) {
				$id = $content['tagId'];
			}
			elseif ( $content['smartTagId'] ) {
				$id = $content['smartTagId'];
			}

			if ( isset( $id ) && ! empty( $id ) ) {
				$sql     = $wpdb->prepare(
					"SELECT term_id FROM $wpdb->termmeta WHERE `meta_value` = %s",
					$id
				);
				$term    = $wpdb->get_row( $sql, ARRAY_A );
				$term_id = (int) $term['term_id'];

				if ( $term_id ) {
					$taxonomy  = get_term( $term_id, '', ARRAY_A );
					$term_args = [ 'description' => $content['newDesc'] ];

					if ( $content['newAlias'] ) {
						$term_args['name'] = $content['newAlias'];
					}

					$update_term = wp_update_term( $term_id, $taxonomy['taxonomy'], $term_args );

					if ( is_wp_error( $update_term ) ) {
						$this->write_log(
							'description_is_updated: ' . $update_term->get_error_message()
						);
					}
				}
			}

			status_header( 200 );
			exit( '{"status":"ok"}' );
		}

		if ( ! isset( $content['uid'] ) && empty( $content['uid'] ) ) {
			status_header( 400 );
			exit( '{"status":"ok", "message":"Empty UID"}' );
		}

		//$sql        = $wpdb->prepare( "SELECT post_id FROM $wpdb->postmeta WHERE `meta_key` = %s AND meta_value = %s", $meta_key, $content['uid'] );
		$sql        = $wpdb->prepare(
			"SELECT ID FROM $wpdb->postmeta AS pmeta JOIN $wpdb->posts AS posts ON pmeta.post_id = posts.ID WHERE pmeta.meta_value = %s AND pmeta.meta_key = %s LIMIT 1",
			$content['uid'],
			$meta_key
		);
		$project_id = $wpdb->get_row( $sql, ARRAY_A );
		$id_post    = $project_id['ID'] ?: '';

		if (
			$project['triggerType'] === 'content_is_created' ||
			$project['triggerType'] === 'content_is_updated' ||
			$project['triggerType'] === 'media_is_created' ||
			$project['triggerType'] === 'media_is_updated'
		) {
			ignore_user_abort( true );
			$show_data = '{"status":"ok"}';

			if ( $id_post ) {
				$data_for_export = $this->update_content( $content, $id_post ); // update content
			}
			else {
				$data_for_export = $this->update_content( $content ); //create content
			}

			status_header( 200 );
			header( 'Connection: close' );
			header( 'Content-Length: ' . strlen( $show_data ) );
			echo $show_data;
			ob_end_flush();
			flush();

			if ( ! isset( $data_for_export['success'] ) && isset( $data_for_export['status'] ) ) {
				$this->export_saas_data( $data_for_export );
			}

			exit();
		}

		if (
			$project['triggerType'] === 'stage_is_updated' ||
			$project['triggerType'] === 'stage_is_created' ||
			$project['triggerType'] === 'content-status_is_updated'
		) {
			ignore_user_abort( true );

			if ( $project['actionType'] === 'delete' ) {
				wp_delete_post( $id_post );

				status_header( 200 );
				exit( '{"status":"ok"}' );
			}

			if ( $project['actionType'] === 'add' ) {
				$show_data = '{"status":"ok"}';

				if ( $id_post ) {
					$data_for_export = $this->update_content( $content, $id_post ); // update content
				}
				else {
					$data_for_export = $this->update_content( $content ); //create content
				}

				status_header( 200 );
				header( 'Connection: close' );
				header( 'Content-Length: ' . strlen( $show_data ) );
				echo $show_data;
				ob_end_flush();
				flush();

				if ( ! isset( $data_for_export['success'] ) && isset( $data_for_export['status'] ) ) {
					$this->export_saas_data( $data_for_export );
				}

				exit();
			}
		}

		if ( $project['triggerType'] === 'content_is_deleted' && $id_post ) {
			wp_delete_post( $id_post );

			if ( 'application/json' === $content_type ) {
				return new \WP_REST_Response( 'ok', 200, '' );
			}

			status_header( 200 );
			exit( '{"status":"ok"}' );
		}
	}

	private function update_content( $content, $update_id = '' ) {
		global $wpdb;

		$prefix         = self::get_metabox_prefix(); //iu-
		$type_prefix    = self::get_type_prefix(); //iu
		$taxonomy_cat   = $type_prefix . '_categories';
		$taxonomy_label = $type_prefix . '_label_tags';
		$taxonomy_tag   = $type_prefix . '_tags';
		$is_blog        = false;

		if ( isset( $content['accountId'] ) && ! empty( $content['accountId'] ) ) {
			update_option( $type_prefix . '_saas_accountId', $content['accountId'], false );
		}

		if ( ucfirst( $content['contentType']['name'] ) === 'Blog' ) {
			if ( $get_blog_type = get_option( '_whither_blog_type' ) ) {
				$is_blog      = true;
				$content_type = $get_blog_type;
			}
		}

		if ( ! $is_blog ) {
			$content_type = self::get_content_type( $content['contentType']['id'], 'id', 'slug' );

			if ( empty( $content_type ) ) {
				$post_type_arr = $this->import_type_posts( $content['contentType'], true, false );
				$type_arr      = end( $post_type_arr );
				$content_type  = $type_arr['slug'];
			}
		}

		$title_and_content = $this->get_title_and_content( $content );
		$project_data      = [
			'post_type'    => $content_type ?: 'post',
			'post_status'  => 'publish',
			'post_excerpt' => $content['internalNote'] ?? '',
			'post_title'   => strip_tags( $title_and_content['title'] ),
			'post_content' => $title_and_content['content'],
			'meta_input'   => $this->import_metapost( $content, $prefix ),
		];

		if ( $is_blog && ( $blog_author = carbon_get_theme_option( $type_prefix . '_blog_author' ) ) ) {
			$project_data['post_author'] = $blog_author[0]['id'];
		}
		else {
			$project_data['post_author'] = get_current_user_id() ?: 1;
		}

		if ( isset( $content['seo']['permalink'] ) && ! empty( $content['seo']['permalink'] ) ) {
			$project_data['post_name'] = sanitize_title( $content['seo']['permalink'] );
		}

		if ( $update_id ) {
			$project_data['ID'] = $update_id;
			$post_id            = wp_update_post( $project_data, true );
		}
		else {
			$post_id = wp_insert_post( $project_data, true );
		}

		if (
			( $citypage_id = (int) get_option( '_template_citypage_id', false ) ) &&
			$content_type === 'iu_city_page'
		) {
			$template_arr     = get_option( '_template_for_content', [] );
			$template_for_cnt = get_post_meta( $citypage_id, '_template_for_content', true ) ?: [];

			if ( ! $template_for_cnt ) {
				$template_for_cnt['iu_city_page'] = $citypage_id;

				if ( ! $template_arr ) {
					$template_arr = $template_for_cnt;
				}
			}
			elseif ( $template_for_cnt && ! isset( $template_for_cnt['iu_city_page'] ) ) {
				$template_for_cnt['iu_city_page'] = $citypage_id;

				if ( ! isset( $template_arr['iu_city_page'] ) ) {
					$template_arr['iu_city_page'] = $citypage_id;
				}
			}

			update_option( '_template_for_content', $template_arr );
			update_post_meta( $citypage_id, '_template_for_content', $template_for_cnt );
		}

		if ( ! is_wp_error( $post_id ) ) {
			if ( $is_blog && get_option( '_iu_duplicate_categories' ) ) {
				$taxonomy_cat = 'category';
				$taxonomy_tag = 'post_tag';
			}

			if ( isset( $content['businessCategories'], $content['businessServices'] ) ){
				$import_cat = $this->import_category(
					$content['businessCategories'],
					$content['businessServices'],
					$taxonomy_cat
				);
				wp_set_object_terms( $post_id, $import_cat, $taxonomy_cat );
			}

			$import_tags = [];
			if ( isset( $content['businessLabels'] ) ){
				$import_tags = $this->import_tags(
					$content['businessLabels'],
					$content['businessServices']['labels'] ?: []
				);
				wp_set_object_terms( $post_id, $import_tags['ids'], $taxonomy_tag );

				$import_label_tags = $this->import_label_tags( $content['businessLabels'] );
				wp_set_object_terms( $post_id, $import_label_tags['ids'], $taxonomy_label );
			}

			if ( isset( $content['smartTags'] ) ){
				$smart_tags = $this->import_smart_tags( $content['smartTags'], $taxonomy_tag );
				wp_set_object_terms( $post_id, $smart_tags, $taxonomy_tag, true );
			}

			if ( ! empty( $content['media'] ) ) {
				$this->import_media(
					$wpdb,
					$post_id,
					$content['media'],
					$content['testimonial'] ?? null,
					$content['presentation'] ?? null
				);
			}

			if (
				! isset( $content['testimonial']['cover']['uid'] ) &&
				empty( $content['testimonial']['cover']['uid'] ) &&
				isset( $content['testimonial']['video']['thumb'] ) &&
				! empty( $content['testimonial']['video']['thumb'] )
			) {
				$this->import_thumb_video( $wpdb, $post_id, $content['testimonial'] );
			}

			set_transient( $type_prefix . '_post_changed', true, MINUTE_IN_SECONDS * 3 );

			$default_display_name = 'Empty';
			$current_user_id      = get_current_user_id();

			if ( $default_user = get_user_by( 'id', $current_user_id ) ) {
				$default_display_name = property_exists( $default_user->data, 'display_name' )
					? $default_user->data->display_name
					: 'Empty';
			}
			else {
				$admins = get_users( [
					'role'    => 'administrator',
					'orderby' => 'ID',
					'order'   => 'ASC',
				] );

				if ( ! empty( $admins ) ) {
					$first_admin          = $admins[0];
					$default_display_name = property_exists( $first_admin, 'display_name' )
						? $first_admin->display_name
						: 'Empty';
				}
			}

			return [
				'status'         => 'published',
				'uid'            => $content['uid'],
				'title'          => $project_data['post_title'],
				'url'            => get_permalink( $post_id ),
				'accountId'      => $content['accountId'] ?? '',
				'contentCreator' => $default_display_name,
				'contentStatus'  => $content['status']['publish'] ?? '',
				'contentType'    => $content['contentType'],
				'contentWord'    => $project_data['post_content'],
				//'date' => date( 'Y-m-d' ) . 'T' . date( 'H:i:s.Z' ) . 'Z',
				'date'           => date( DATE_RFC822 ),
				'time_created'   => get_post_time( 'U', '', $post_id ),
				'time_update'    => $content['updateDate']['_seconds'] ?? '',
				'tags'           => $import_tags['to_export'] ?? '',
			];
		}

		$error = [
			'success' => false,
			'message' => $post_id->get_error_message(),
			'content' => $content,
		];
		$this->write_log( date( 'Y-m-d H:i:s P' ) . ':__ ' . json_encode( $error ), 'error-import.txt' );

		return $error;
	}
}
